<?php
/**
 * Reactions functions
 *
 * @package reactions
 * @version 1.0.0
 * @author Rootali / Slaweally
 * @copyright Copyright (c) 2025 Rootali / Slaweally
 * @license BSD
 */

defined('COT_CODE') or die('Wrong URL');

// Resource’ları yüklüyoruz, böylece $R['reactions_button'] gerçek HTML döner
require_once cot_incfile('reactions', 'plug', 'resources');


// -------------------------------------------------
// Reaction icons fallback (FontAwesome örneği)
// -------------------------------------------------
global $R;
$defaultReactionIcons = [
    'like'  => '<i class="fa fa-thumbs-up"></i>',
    'love'  => '<i class="fa fa-heart"></i>',
    'laugh' => '<i class="fa fa-laugh"></i>',
    'wow'   => '<i class="fa fa-surprise"></i>',
    'sad'   => '<i class="fa fa-sad-tear"></i>',
    'angry' => '<i class="fa fa-angry"></i>'
];
foreach ($defaultReactionIcons as $type => $iconHtml) {
    $key = 'reaction_' . $type . '_icon';
    if (empty($R[$key])) {
        $R[$key] = $iconHtml;
    }
}

// -------------------------------------------------


// Kesin tablo kaydı için:
global $db;
$db->registerTable('reactions');


// Define realm for caching
define('REACTIONS_REALM', 'reactions');

// Constants for error codes
define('REACTION_ERR_NONE', 0);
define('REACTION_ERR_MAXDAY', 1);
define('REACTION_ERR_MAXUSER', 2);
define('REACTION_ERR_ITEM', 3);
define('REACTION_ERR_SELF', 4);

/**
 * Add a new reaction
 * 
 * @param int $to_user Reaction receiver ID
 * @param int $from_user Reaction sender ID
 * @param string $ext Extension code
 * @param int $item Item ID
 * @param string $type Reaction type
 * @return bool
 */
function reaction_add($to_user, $from_user, $ext, $item, $type) {
    Cot::$cache && Cot::$cache->clear_realm(REACTIONS_REALM, COT_CACHE_TYPE_ALL);
    $db_reactions = Cot::$db->reactions;
    $db_users = Cot::$db->users;

    // First try to update an existing reaction
    $updated = Cot::$db->update($db_reactions, 
        array(
            'reaction_type' => $type,
            'reaction_date' => Cot::$sys['now']
        ),
        "reaction_from_user = ? AND reaction_ext = ? AND reaction_item = ?",
        array($from_user, $ext, $item)
    );

    // If no rows were updated, insert a new one
    if (!$updated) {
        $ins = Cot::$db->insert($db_reactions, array(
            'reaction_date' => Cot::$sys['now'],
            'reaction_to_user' => $to_user,
            'reaction_from_user' => $from_user,
            'reaction_ext' => $ext,
            'reaction_item' => $item,
            'reaction_type' => $type
        ));
        
        if ($ins) {
            /* === Hook === */
            foreach (cot_getextplugins('reactions.add.done') as $pl) {
                include $pl;
            }
            /* ===== */

            // Update user reactions count
            Cot::$db->query("UPDATE $db_users SET user_reactions_received = user_reactions_received + 1 WHERE user_id = ?", array($to_user));
            Cot::$db->query("UPDATE $db_users SET user_reactions_given = user_reactions_given + 1 WHERE user_id = ?", array($from_user));
            
            return true;
        }
        return false;
    }
    return true;
}

/**
 * Remove a reaction
 * 
 * @param int $from_user User ID who gave the reaction
 * @param string $ext Extension code
 * @param int $item Item ID
 * @return bool
 */
function reaction_remove($from_user, $ext, $item) {
    Cot::$cache && Cot::$cache->clear_realm(REACTIONS_REALM, COT_CACHE_TYPE_ALL);
    $db_reactions = Cot::$db->reactions;
    $db_users = Cot::$db->users;
    
    // Get the reaction details first
    $reaction = Cot::$db->query("SELECT * FROM $db_reactions WHERE reaction_from_user = ? AND reaction_ext = ? AND reaction_item = ?", 
        array($from_user, $ext, $item))->fetch();
    
    if ($reaction) {
        $to_user = $reaction['reaction_to_user'];
        
        // Delete the reaction
        $rm = Cot::$db->delete($db_reactions, 
            "reaction_from_user = ? AND reaction_ext = ? AND reaction_item = ?", 
            array($from_user, $ext, $item));
        
        if ($rm) {
            // Update user reaction counts
            Cot::$db->query("UPDATE $db_users SET user_reactions_received = GREATEST(0, user_reactions_received - 1) WHERE user_id = ?", array($to_user));
            Cot::$db->query("UPDATE $db_users SET user_reactions_given = GREATEST(0, user_reactions_given - 1) WHERE user_id = ?", array($from_user));
            
            return true;
        }
    }
    
    return false;
}

/**
 * Check if it is correct to add a new reaction
 *
 * @param int $to_user Reaction receiver ID
 * @param int $from_user Reaction sender ID
 * @param string $ext Extension code
 * @param int $item Item ID
 * @return int One of the REACTION_ERR_* constants
 */
function reaction_check($to_user, $from_user, $ext, $item) {
    $db_reactions = Cot::$db->reactions;

    // Can't react to yourself
    if ($to_user == $from_user) {
        return REACTION_ERR_SELF;
    }

    // Check daily limit
    if (Cot::$db->query("SELECT COUNT(*) FROM $db_reactions 
            WHERE reaction_from_user = ? AND DATE(FROM_UNIXTIME(reaction_date)) = DATE(NOW())", 
            array($from_user))->fetchColumn() >= Cot::$cfg['plugin']['reactions']['maxday']) {
        return REACTION_ERR_MAXDAY;
    }

    // Check per-user daily limit
    if (Cot::$db->query("SELECT COUNT(*) FROM $db_reactions 
            WHERE reaction_from_user = ? AND reaction_to_user = ? AND DATE(FROM_UNIXTIME(reaction_date)) = DATE(NOW())", 
            array($from_user, $to_user))->fetchColumn() >= Cot::$cfg['plugin']['reactions']['maxuser']) {
        return REACTION_ERR_MAXUSER;
    }

    // User can change their reaction, so we don't need to check for existing reactions
    return REACTION_ERR_NONE;
}

/**
 * Check if a user has already reacted to an item
 * 
 * @param int $from_user User ID
 * @param string $ext Extension code
 * @param int $item Item ID
 * @return string|bool Reaction type if exists, false otherwise
 */
function reaction_get_user_reaction($from_user, $ext, $item) {
    $db_reactions = Cot::$db->reactions;
    $reaction = Cot::$db->query("SELECT reaction_type FROM $db_reactions 
        WHERE reaction_from_user = ? AND reaction_ext = ? AND reaction_item = ?", 
        array($from_user, $ext, $item))->fetchColumn();
    
    return $reaction ?: false;
}

/**
 * Get reactions count for an item
 * 
 * @param string $ext Extension code
 * @param int $item Item ID
 * @param string $type Optional reaction type filter
 * @return int Number of reactions
 */
function reaction_count($ext, $item, $type = '') {
    $db_reactions = Cot::$db->reactions;
    
    $params = array($ext, $item);
    $where = "reaction_ext = ? AND reaction_item = ?";
    
    if (!empty($type)) {
        $where .= " AND reaction_type = ?";
        $params[] = $type;
    }
    
    return Cot::$db->query("SELECT COUNT(*) FROM $db_reactions WHERE $where", $params)->fetchColumn();
}

/**
 * Get reactions by type for an item
 * 
 * @param string $ext Extension code
 * @param int $item Item ID
 * @return array Associative array of reaction types and counts
 */
function reaction_get_counts($ext, $item) {
    $db_reactions = Cot::$db->reactions;
    
    $result = array(
        'like' => 0,
        'love' => 0,
        'laugh' => 0,
        'wow' => 0,
        'sad' => 0,
        'angry' => 0,
        'total' => 0
    );
    
    $res = Cot::$db->query("SELECT reaction_type, COUNT(*) as count FROM $db_reactions 
        WHERE reaction_ext = ? AND reaction_item = ? 
        GROUP BY reaction_type", array($ext, $item));
    
    $total = 0;
    while ($row = $res->fetch()) {
        $result[$row['reaction_type']] = (int)$row['count'];
        $total += (int)$row['count'];
    }
    
    $result['total'] = $total;
    return $result;
}

/**
 * Get users who reacted to an item
 * 
 * @param string $ext Extension code
 * @param int $item Item ID
 * @param string $type Optional reaction type filter
 * @param int $limit Maximum number of users to return
 * @return array Array of user data
 */
function reaction_get_users($ext, $item, $type = '', $limit = 0) {
    $db_reactions = Cot::$db->reactions;
    $db_users = Cot::$db->users;
    
    $params = array($ext, $item);
    $where = "r.reaction_ext = ? AND r.reaction_item = ?";
    
    if (!empty($type)) {
        $where .= " AND r.reaction_type = ?";
        $params[] = $type;
    }
    
    $limit_sql = $limit > 0 ? " LIMIT " . (int)$limit : "";
    
    $res = Cot::$db->query("SELECT r.*, u.user_name 
        FROM $db_reactions AS r
        LEFT JOIN $db_users AS u ON u.user_id = r.reaction_from_user
        WHERE $where
        ORDER BY r.reaction_date DESC
        $limit_sql", $params);
    
    $users = array();
    while ($row = $res->fetch()) {
        $users[] = array(
            'user_id' => $row['reaction_from_user'],
            'user_name' => $row['user_name'],
            'reaction_type' => $row['reaction_type'],
            'reaction_date' => $row['reaction_date']
        );
    }
    
    return $users;
}

/**
 * Generate HTML for reaction buttons
 * 
 * @param string $ext Extension code
 * @param int $item Item ID
 * @param int $owner_id Owner user ID
 * @param string $css_class CSS class
 * @return string HTML code
 */
function reaction_buttons($ext, $item, $owner_id, $css_class = '') {
    global $usr, $L, $R;
    
    if (empty($usr['id']) || $usr['id'] == $owner_id) {
        return '';  // Guest users or content owners can't react
    }
    
    $current_reaction = reaction_get_user_reaction($usr['id'], $ext, $item);
    $html = '<div class="reactions-container">';
    
    // Reaction types from config
    $types = array(
        'like' => Cot::$cfg['plugin']['reactions']['like'],
        'love' => Cot::$cfg['plugin']['reactions']['love'],
        'laugh' => Cot::$cfg['plugin']['reactions']['laugh'],
        'wow' => Cot::$cfg['plugin']['reactions']['wow'],
        'sad' => Cot::$cfg['plugin']['reactions']['sad'],
        'angry' => Cot::$cfg['plugin']['reactions']['angry']
    );
    
    foreach ($types as $type => $enabled) {
        if ($enabled) {
            $icon = $R['reaction_' . $type . '_icon'];
            $active = ($current_reaction === $type) ? 'active' : '';
            
            $html .= cot_rc('reactions_button', array(
                'class' => $css_class,
                'active' => $active,
                'type' => $type,
                'ext' => $ext,
                'item' => $item,
                'icon' => $icon
            ));
        }
    }
    
    $html .= '</div>';
    return $html;
}

/**
 * Generate HTML for reaction counts display
 * 
 * @param string $ext Extension code
 * @param int $item Item ID
 * @return string HTML code
 */
function reaction_counts_display($ext, $item) {
    global $R;
    
    if (!Cot::$cfg['plugin']['reactions']['display_count']) {
        return '';
    }
    
    $counts = reaction_get_counts($ext, $item);
    if ($counts['total'] == 0) {
        return '';
    }
    
    $html = '<div class="reactions-counts">';
    
    // Reaction types from config
    $types = array(
        'like' => Cot::$cfg['plugin']['reactions']['like'],
        'love' => Cot::$cfg['plugin']['reactions']['love'],
        'laugh' => Cot::$cfg['plugin']['reactions']['laugh'],
        'wow' => Cot::$cfg['plugin']['reactions']['wow'],
        'sad' => Cot::$cfg['plugin']['reactions']['sad'],
        'angry' => Cot::$cfg['plugin']['reactions']['angry']
    );
    
    foreach ($types as $type => $enabled) {
        if ($enabled && $counts[$type] > 0) {
            $icon = $R['reaction_' . $type . '_icon'];
            
            if (Cot::$cfg['plugin']['reactions']['display_users'] && Cot::$cfg['plugin']['reactions']['use_tooltips']) {
                $users = reaction_get_users($ext, $item, $type, Cot::$cfg['plugin']['reactions']['maxreacted']);
                $user_names = array();
                foreach ($users as $user) {
                    $user_names[] = htmlspecialchars($user['user_name']);
                }
                $tooltip_text = implode(', ', $user_names);
                if (count($users) < $counts[$type]) {
                    $tooltip_text .= ' and ' . ($counts[$type] - count($users)) . ' more';
                }
                
                $html .= '<span class="reaction-stat">' . $icon . ' ' . 
                        cot_rc('reactions_tooltip', array(
                            'count' => $counts[$type],
                            'users' => $tooltip_text,
                            'type' => $type,
                            'ext' => $ext,
                            'item' => $item
                        )) . '</span>';
            } else {
                $html .= '<span class="reaction-stat">' . $icon . ' ' . 
                        cot_rc('reactions_count', array(
                            'count' => $counts[$type],
                            'type' => $type,
                            'ext' => $ext,
                            'item' => $item
                        )) . '</span>';
            }
        }
    }
    
    $html .= '</div>';
    return $html;
}